/*
 * libclp, a library to parse command line and rc files
 * Copyright (C) 2002 Olivier Mehani <shtrom@ssji.net>
 * 
 * $Id: clp.c,v 1.2 2002/08/20 05:39:23 shtrom Exp $
 * command line parsing functions
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <stdlib.h>
#include "clp.h"

static ClpOpt *clp_guess_opt PARAMS((char *arg, ClpOpt *opts, int size));
static int clp_opt_type PARAMS((char *arg));
static ClpOpt *clp_guess_short_opt PARAMS((char *arg, ClpOpt *opts, int size));
static ClpOpt *clp_guess_long_opt PARAMS((char *arg, ClpOpt *opts, int size));

int clp_parse_opts(int argc, char **argv, ClpOpt * opts, int size)
{
	int i, j, len;
	char tmp[2];
	ClpOpt *opt;

	for (i = 1; i < argc; i++) {
		opt = clp_guess_opt(argv[i], opts, size);
		if (!opt)
			return i;

		switch (opt->opt_type) {
		case OPT_SINGLE:
			*(int *) opt->var = TRUE;

			if (opt->unique)
				return 0;
			break;
		case OPT_CHAR:
			i++;

			if (i < argc && clp_opt_type(argv[i]) == OPT_TYPE_UNKNOWN)
				*(char *) opt->var = argv[i][0];
			else
				return -(i - 1);

			if (opt->unique)
				return 0;
			break;
		case OPT_CHAR_LIST:
			j = i++;

			while (++j < argc
				   && clp_opt_type(argv[j]) == OPT_TYPE_UNKNOWN);

			if (j > i) {
				len = j - i;

				*(char **) opt->var = malloc((len + 1) * sizeof(char));
				**(char **) opt->var = 0;

				for (j = 0; j < len; j++) {
					snprintf(tmp, 2, "%c", *argv[i + j]);
					strcat(*(char **) opt->var, tmp);
				}
				i += --j;
			} else
				return -(i - 1);

			if (opt->unique)
				return 0;
			break;
		case OPT_STRING:
			i++;

			if (i < argc && clp_opt_type(argv[i]) == OPT_TYPE_UNKNOWN)
				*(char **) opt->var = argv[i];
			else
				return -(i - 1);

			if (opt->unique)
				return 0;
			break;
		case OPT_STRING_LIST:
			j = i++;

			while (++j < argc
				   && clp_opt_type(argv[j]) == OPT_TYPE_UNKNOWN);

			if (j > i) {
				len = j - i;

				*(char ***) opt->var = malloc((len + 1) * sizeof(char *));

				*((*(char ***) opt->var) + len) = NULL;

				for (j = 0; j < len; j++)
					*((*(char ***) opt->var) + j) = argv[i + j];

				i += --j;
			} else
				return -(i - 1);

			if (opt->unique)
				return 0;
			break;
		case OPT_INT:
			i++;

			if (i < argc && clp_opt_type(argv[i]) == OPT_TYPE_UNKNOWN)
				*(int *) opt->var = atoi(argv[i]);
			else
				return -(i - 1);

			if (opt->unique)
				return 0;
			break;
		case OPT_INT_LIST:
			j = i++;

			while (++j < argc
				   && clp_opt_type(argv[j]) == OPT_TYPE_UNKNOWN);

			if (j > i) {
				len = j - i;
				*(int **) opt->var = malloc((len + 1) * sizeof(int));
				*(*(int **) opt->var + len) = -1;

				for (j = 0; j < len; j++)
					*(*(int **) opt->var + j) = atoi(argv[i+j]);

				i += --j;
			} else
				return -(i - 1);

			if (opt->unique)
				return 0;
			break;
		case OPT_FLOAT:
			i++;

			if (i < argc && clp_opt_type(argv[i]) == OPT_TYPE_UNKNOWN)
				*(float *) opt->var = atof(argv[i]);
			else
				return -(i - 1);

			if (opt->unique)
				return 0;
			break;
		case OPT_FLOAT_LIST:
			j = i++;

			while (++j < argc
				   && clp_opt_type(argv[j]) == OPT_TYPE_UNKNOWN);

			if (j > i) {
				len = j - i;
				*(float **) opt->var = malloc((len + 1) * sizeof(float));
				*(*(float **) opt->var + len) = -1;

				for (j = 0; j < len; j++)
					*(*(float **) opt->var + j) = atof(argv[i+j]);

				i += --j;
			} else
				return -(i - 1);

			if (opt->unique)
				return 0;
			break;
		default:
			break;
		}
	}
	return 0;
}

ClpOpt *clp_guess_opt(char *arg, ClpOpt * opts, int size)
{
	int opt_type;

	opt_type = clp_opt_type(arg);

	switch (opt_type) {
	case OPT_TYPE_SHORT:
		return clp_guess_short_opt(arg, opts, size);
		break;
	case OPT_TYPE_LONG:
		return clp_guess_long_opt(arg, opts, size);
		break;
	default:
		return NULL;
		break;
	}

	return NULL;
}

int clp_opt_type(char *arg)
{
	if (arg[0] != '-')
		return OPT_TYPE_UNKNOWN;
	else {
		if (arg[1] != '-') {
			return OPT_TYPE_SHORT;
		} else
			return OPT_TYPE_LONG;
	}

	return -1;
}

ClpOpt *clp_guess_short_opt(char *arg, ClpOpt * opts, int size)
{
	int i;

	for (i = 0; i < size && opts[i].sht_opt != arg[1]; i++);

	if (i == size && opts[i].sht_opt != arg[1])
		return NULL;

	return &opts[i];
}

ClpOpt *clp_guess_long_opt(char *arg, ClpOpt * opts, int size)
{
	int i;

	//why size-1 ?!
	for (i = 0; i < size - 1 && strcmp(opts[i].lng_opt, &arg[2]); i++);

	if (i == size - 1 && strcmp(opts[i].lng_opt, &arg[2]))
		return NULL;

	return &opts[i];
}

char *clp_gen_help(ClpOpt * opts, int size, char *name)
{
	char *helpstring, *tmp, *usagestring = "Usage: %s [OPTIONS...]\n";
	int i, tmplen, len = 0;

	len = strlen(usagestring) - 2 + strlen(name);
	for (i = 0; i < size; i++) {
		len += 2;
		if (opts[i].sht_opt) {
			len += 2;
			if (opts[i].lng_opt)
				len += 2;
		}
		if (opts[i].lng_opt)
			len += strlen(opts[i].lng_opt) + 2;
		len += 2;
		if (opts[i].hlp_msg)
			len += strlen(opts[i].hlp_msg);
		len += 1;
	}

	helpstring = malloc(len);

	snprintf(helpstring, len, usagestring, name);
#define addhs(hsst) strncat(helpstring, hsst,len-strlen(helpstring))

	for (i = 0; i < size; i++) {
		addhs("  ");
		if (opts[i].sht_opt) {
			tmp = malloc(3);
			snprintf(tmp, 3, "-%c", opts[i].sht_opt);
			addhs(tmp);
			free(tmp);
			if (opts[i].lng_opt)
				addhs(", ");
		}
		if (opts[i].lng_opt) {
			tmplen = strlen(opts[i].lng_opt) + 3;
			tmp = malloc(tmplen);
			snprintf(tmp, tmplen, "--%s", opts[i].lng_opt);
			addhs(tmp);
			free(tmp);
		}
		addhs("\t\t");
		if (opts[i].hlp_msg)
			addhs(opts[i].hlp_msg);
		addhs("\n");
	}

	return helpstring;
#undef addhs
}

void clp_free_help(char *helpstring)
{
	free(helpstring);
}
